/* $Id: pbck.c,v 1.18 1998/08/11 23:30:19 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Bryan Murray, updated by Eric Backus */

/* pbck.c
 *
 * To get a usage message summarizing the command-line options for
 * this program, use "./pbck -u".
 *
 * This file demonstrates how a previously recorded file from the E1432 can
 * be post-processed.  It is expected that the E1562 LIF library, and SICL
 * be installed on the host machine.  The previously recorded file must
 * use the same header at the beginning of the file as this program is
 * expecting in order to correctly interpret the results.
 *
 * This program reads the header from the specified file and determines
 * the recorded blocksize, channel count, scaling, etc. and then creates
 * an xplot window for each channel in the file and refreshes the window
 * with new data once for each scan.  The playback blocksize may be specified
 * as different from the recorded blocksize.  Type 'pbck -u' to see options
 * on how to use the program.
 *
 * WARNING:  This example may NOT be able to post-process a file which
 * has been copied to another file on the LIF volume or copied to the
 * host.  The reason is that the E1562 sequence depends upon the data
 * being located at the beginning of SCSI block.  It is possible to make
 * the sequences deal with this, but it requires additional coding.  One
 * work-around is to add code to determine if the beginning of the data
 * really does align with the beginning of a SCSI block -- if not start
 * the sequence at the previous SCSI block and throw away the bytes
 * preceeding the data.  A similar adjustment may need to be done at the
 * end of the data.  Another possible work-around would be to try chaining
 * one sequence to another, where the first sequence just reads the
 * initial bytes before the data, and the second sequence is the same as
 * the one which is now defined for playback in e1562tput.c.
 *
 * Another problem which has not been solved is that of starting a playback
 * somewhere in the middle of the data stream.  One way to solve this is
 * to add code to the beginning of readPlaybackData to just read data up
 * to the point where it is desired to start post-processing.  The down
 * side of this method is that it could take a while for very large
 * throughput files.  The second method would be to calculate where a scan
 * begins in the file and back up to the previous SCSI block to start the
 * sequence and discard the initial partial scan.
 */

#include "tputhdr.h"    /* shared header between record and playback */
#include "e1562tput.h"  /* includes e1562lif.h */
#include "xplot.h"
#include <e1432.h>      /* need data structure definitions and macros */
#include <sicl.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>


#define	DEFAULT_AUTOSCALE		0
#define DEFAULT_INTERFACE               "vxi"
#define DEFAULT_E1562_LOGICAL_ADDRESS   144
#define DEFAULT_NUMBER_SCANS            10
#define	DEFAULT_DISPLAY_TRAILERS	0
#define DEFAULT_BLOCKSIZE               2048
#define DEFAULT_DEBUG_LEVEL             0
#define	DEFAULT_WAIT_WHEN_DONE		1


/* XPLOT information
 */
#define PLOTS_X 4
#define PLOTS_Y 6
#define PLOT_X_SIZE 256
#define PLOT_X_EXTRA 20
#define PLOT_Y_SIZE 128
#define PLOT_Y_EXTRA 42
#define PLOT_Y_RANGE 1.0




/* This function will print every field in the header.  It is primarily
 * for debug purposes to verify that what is getting written to the
 * throughput file is what is getting read from the throughput file.
 */
static void printHeader(const char* filename,
                        const InputHwInformation* inputSetup)
{
    short index;

    printf("%s  throughput file information:\n", filename);
    printf("    total channels = %d\n", inputSetup->channelCount);
    printf("    recorded channels = %d\n", inputSetup->activeChanCount);
    printf("    data offset = %lu\n", inputSetup->dataOffset);
    printf("    first scan offset = %lu\n", inputSetup->firstScanOffset);
    printf("    samples per block = %lu\n", inputSetup->samplesPerBlock);
    printf("    number of scans = %lu\n", inputSetup->scanCount);
    printf("    data size enum = %d\n", inputSetup->dataSize);
    printf("    data mode enum = %d\n", inputSetup->dataMode);
    printf("    trailer enabled = %d\n", inputSetup->appendStatusEnabled);
    printf("    span = %g\n", inputSetup->span);
    printf("    zoom enabled = %d\n", inputSetup->zoomEnabled);
    printf("    center freq = %g\n", inputSetup->centerFreq);
    for (index = 0; index < MAX_MODULES * MAX_CHANNELS_PER_MODULE; ++index)
    {
        if (inputSetup->scaleFactor[index] != 0.0)
            printf("    channel %3d scale factor = %g\n",
                   index + 1, inputSetup->scaleFactor[index]);
    }

    printf("\n");
}


/* Parse the address string into a list of numbers representing logical
 * addresses.  A comma is used as a number seperator, and a colon is used
 * to specify a range from left number to right number including both
 * numbers.  Return the array of logical addresses and the number of
 * addresses found.  If any error is detected in the string, the returned
 * count will be 0.  Possible errors include a bad character found, the
 * addresses found are not in numerical order, an unterminated range was
 * discovered.
 */
static void parseAddressList(const char* address, short* array,
                             short* arrayCount, short arraySize)
{
    const char* s;
    short index;
    int lastVal = -1;
    int val;

    /* Check for valid parameters */
    if (array == 0 && arrayCount == 0) return;
    if (arrayCount != 0) *arrayCount = 0;
    if (arraySize == 0 || address == 0 || *address == '\0') return;

    /* Loop through the address list looking for numbers */
    for (s = address, index = 0; *s ; )
    {
        /* Skip whitespace */
        while (*s == ' ' || *s == '\t') ++s;
        if (*s == '\0') break;

        /* Parse the next logical address */
        if (!isdigit(*s)) return /* error */;
        val = atoi(s);
        if (val <= lastVal) return /* error */;
        lastVal = val;
        while (isdigit(*s)) ++s;

        /* Put the logical address into the array */
        if (index < arraySize && array != 0) array[index] = (short)val;
        ++index;

        /* Skip whitespace */
        while (*s == ' ' || *s == '\t') ++s;

        /* Check for a range of logical addresses */
        if (*s == ':')
        {
            int val2;

            ++s;

            /* Skip whitespace */
            while (*s == ' ' || *s == '\t') ++s;

            /* Parse the ending range of logical addresses. */
            if (!isdigit(*s)) return /* error */;
            val2 = atoi(s);
            while (isdigit(*s)) ++s;
            if (val2 < val) return /* error */;

            /* Put the list of logical addresses into the returned array */
            for (++val; val <= val2; ++val)
                if (index < arraySize && array != 0)
		    array[index++] = (short)val;
            lastVal = val2;
        }

        /* Skip whitespace and logical address seperator */
        while (*s == ' ' || *s == '\t') ++s;
        if (*s == ',') ++s;
    }

    /* Set the return count of logical addresses found. */
    if (arrayCount != 0) *arrayCount = index;
}


/* Return the number of bytes per sample given a particular data_size.
 */
static short dataSizeBytes(short dataSize)
{
    /* What about complex data? */

    switch (dataSize)
    {
    case E1432_DATA_SIZE_16: return 2;
    case E1432_DATA_SIZE_32: return 4;
    case E1432_DATA_SIZE_32_SERV: return 4;
    case E1432_DATA_SIZE_FLOAT32: return 4;
    default: return 0;
    }
}


/* Initialize the xplot windows.
 */
static void plotInitialize(const InputHwInformation* inputSetup,
                           short* channelList, short channelCount,
                           long blocksize, float* dataBuf,
                           char** plotID, int autoscale)
{
    int i, x, y;
    char geometry[32], title[40];
    short chanNum;
    float ymax;

    chanNum = 0;
    for (i = 0; i < channelCount;  ++i)
    {
        chanNum = channelList[i];

	plotID[i] = NULL;

        switch (inputSetup->dataSize)
        {
        case E1432_DATA_SIZE_16:
	    /* The 2.1 is a magic number from the E1432 firmware
	       (digital filter safe scaling constant).  The number is
	       1.2 for the E1433 module.  It would be better for the
	       thruput file to record what the range was, so we could
	       just use that. */
	    ymax = 32768.0 * inputSetup->scaleFactor[chanNum] / 2.1;
	    break;
	case E1432_DATA_SIZE_32:
	case E1432_DATA_SIZE_32_SERV:
	    ymax = 2147483648.0 * inputSetup->scaleFactor[chanNum] / 2.1;
	    break;
	default:
	    ymax = 10.0;	/* We can't tell what the max was */
	    break;
	}

	/* locations of the displays on the screen */
	x = (PLOT_X_SIZE + PLOT_X_EXTRA) * (i % PLOTS_X);
	y = (PLOT_Y_SIZE + PLOT_Y_EXTRA) * (i / PLOTS_X % PLOTS_Y);
	
	(void) sprintf(geometry, "%dx%d+%d+%d",
		       PLOT_X_SIZE, PLOT_Y_SIZE, x, y);
	(void) sprintf(title, "Chan %d", chanNum + 1);

	plotID[i] = xplot_init_plot(&dataBuf[i * blocksize], (int)blocksize,
                                    (float)inputSetup->span * 1000.0,
                                    ymax, -ymax,
                                    TIME_TRACE, geometry, title);

	if (autoscale)
	    xplot_change_yautoscale(plotID[i], 1);

        ++chanNum;
    }
}


/* Read one recorded scan into memory from the E1562 and scale the data.
 */
static long processScan(const InputHwInformation* inputSetup,
                        short* channelList, short channelCount,
                        long samplesToCollect,
                        void* recordScanBuffer, long recordSampleOffset,
                        long bytesPerRecordBlock,
                        float* playbackResultsBuf, long playbackSampleOffset,
                        long playbackBlocksize,
                        long* samplesRead, int display_trailers)
{
    short* sbuf;
    long* lbuf;
    float* fbuf;
    short chan;
    short chanNum;
    long index;
    long samplesCollected = 0;
    long dispIndex;

    sbuf = (short*)recordScanBuffer;
    lbuf = (long*)recordScanBuffer;
    fbuf = (float*)recordScanBuffer;
    chanNum = 0;

    for (chan = 0; chan < (short) inputSetup->activeChanCount; ++chan)
    {
        /* Find the actual channel number for the data */
        for ( ; inputSetup->scaleFactor[chanNum] == 0.0; ++chanNum) ;

	if (display_trailers && inputSetup->appendStatusEnabled != 0)
	{
	    struct e1432_trailer *trailer;

	    trailer = (struct e1432_trailer *) (void *)
		(((char *) sbuf) + bytesPerRecordBlock -
		 sizeof *trailer);
	    (void) printf("Trailer %d:\n", chan);
	    (void) printf(" trig_corr %6g  ", trailer->trig_corr);
	    (void) printf(" zoom_corr %6g  ", trailer->zoom_corr);
	    (void) printf(" rpm1      %6g  ", trailer->rpm1);
	    (void) printf(" rpm2      %6g\n", trailer->rpm2);
	    (void) printf(" gap       %6d  ", trailer->gap);
	    (void) printf(" info  0x%08x  ", trailer->info);
	    (void) printf(" peak      %6g  ", trailer->peak);
	    (void) printf(" rms       %6g\n", trailer->rms);
	}

        /* Determine whether we are supposed to display this channel */
        for (dispIndex = 0; dispIndex < channelCount; ++dispIndex)
            if (channelList[dispIndex] == chanNum) break;
        if (dispIndex >= channelCount)
	{
	    sbuf = (short*)(void *)((char*)sbuf + bytesPerRecordBlock);
	    lbuf = (long*)(void *)((char*)lbuf + bytesPerRecordBlock);
	    fbuf = (float*)(void *)((char*)fbuf + bytesPerRecordBlock);
	    chanNum++;
	    continue;
	}

        switch (inputSetup->dataSize)
        {
        case E1432_DATA_SIZE_16:
            for (index = 0; index < samplesToCollect &&
                 index + recordSampleOffset < inputSetup->samplesPerBlock;
                 ++index)
            {
                playbackResultsBuf[dispIndex * playbackBlocksize + index +
                                   playbackSampleOffset] =
                    sbuf[index + recordSampleOffset] *
                        inputSetup->scaleFactor[chanNum];
            }

            if (samplesCollected == 0) samplesCollected = index;

            break;

        case E1432_DATA_SIZE_32_SERV:
            for (index = 0; index < samplesToCollect &&
                 index + recordSampleOffset < inputSetup->samplesPerBlock;
                 ++index)
            {
                playbackResultsBuf[dispIndex * playbackBlocksize + index +
                                   playbackSampleOffset] =
                    (long)(lbuf[index + recordSampleOffset] & 0xffffff00UL) *
                        inputSetup->scaleFactor[chanNum];
            }

            if (samplesCollected == 0) samplesCollected = index;

            break;

        case E1432_DATA_SIZE_32:
            for (index = 0; index < samplesToCollect &&
                 index + recordSampleOffset < inputSetup->samplesPerBlock;
                 ++index)
            {
                playbackResultsBuf[dispIndex * playbackBlocksize + index +
                                   playbackSampleOffset] =
                    lbuf[index + recordSampleOffset] *
                        inputSetup->scaleFactor[chanNum];
            }

            if (samplesCollected == 0) samplesCollected = index;

            break;

        case E1432_DATA_SIZE_FLOAT32:
            for (index = 0; index < samplesToCollect &&
                 index + recordSampleOffset < inputSetup->samplesPerBlock;
                 ++index)
            {
                playbackResultsBuf[dispIndex * playbackBlocksize + index +
                                   playbackSampleOffset] =
                    fbuf[index + recordSampleOffset];
            }

            if (samplesCollected == 0) samplesCollected = index;

            break;

        default:
            printf("  Data format %d not supported\n", inputSetup->dataSize);
            break;
        }

        sbuf = (short*)(void *)((char*)sbuf + bytesPerRecordBlock);
        lbuf = (long*)(void *)((char*)lbuf + bytesPerRecordBlock);
        fbuf = (float*)(void *)((char*)fbuf + bytesPerRecordBlock);
        ++chanNum;
    }

    *samplesRead = samplesCollected;

    return 0;
}


/* Loop through the specified number of playback scans.  Each scan is read
 * through one or more calls e1562tput_readPlaybackScan and processScan.
 * One playback scan may consist of several record scans, or may be only
 * part of a record scan depending upon the blocksize at record and playback.
 */
static long readPlaybackData(const InputHwInformation* inputSetup,
                             E1562Handle H1562,
                             long numberScans, long playbackBlocksize,
                             short* channelList, short channelCount,
                             long debug, int autoscale,
			     int waitWhenDone, int display_trailers)
{
    void* buf;
    float* dataBuf;
    short bytesPerSample;
    long bytesPerBlock;
    long bytesPerScan;
    long scan;
    long err;
    char** plotID;
    long samplesRead;
    long samplesCollected;
    long chan;
    long recordSample;
    long blockIndex;

    bytesPerSample = dataSizeBytes(inputSetup->dataSize);
    bytesPerBlock = bytesPerSample * inputSetup->samplesPerBlock;
    if (inputSetup->appendStatusEnabled != 0)
    {
        bytesPerBlock += sizeof(struct e1432_trailer);
    }

    bytesPerScan = bytesPerBlock * inputSetup->activeChanCount;

    buf = malloc(bytesPerScan);
    if (buf == 0) return -1;

    dataBuf = (float*)malloc(playbackBlocksize * sizeof(float) * channelCount);
    if (dataBuf == 0) return -5;

    /* allocate enough plotID's for all the channels */
    plotID = (char**)malloc(channelCount * sizeof(char*));
    if (plotID == 0) return -4;

    plotInitialize(inputSetup, channelList, channelCount,
                   playbackBlocksize, dataBuf, plotID, autoscale);

    recordSample = inputSetup->samplesPerBlock + 1;
    blockIndex = 0;
    samplesRead = 0;
    for (scan = 0; scan < numberScans; ++scan)
    {
        for (samplesCollected = 0; samplesCollected < playbackBlocksize;
             samplesCollected += samplesRead)
        {
            if (recordSample >= inputSetup->samplesPerBlock)
            {
                for (blockIndex = 0;
		     blockIndex < (long) inputSetup->activeChanCount;
                     ++blockIndex)
                {
                    err = e1562tput_readPlaybackScan
                        (H1562, bytesPerBlock, &((char*)buf)[blockIndex *
                                                            bytesPerBlock]);
                    if (err != 0) return err;
                }

                recordSample = 0;
            }

            if (debug > 10)
            printf("scan = %ld, samplesCollected = %ld, samplesRead = %ld, "
                   "recordSample = %ld\n",
                   scan, samplesCollected, samplesRead, recordSample);

            err = processScan(inputSetup,
                              channelList, channelCount,
                              playbackBlocksize - samplesCollected,
                              buf, recordSample, bytesPerBlock,
                              dataBuf, samplesCollected, playbackBlocksize,
                              &samplesRead, display_trailers);
            if (err != 0) return err;

            recordSample += samplesRead;
        }

        for (chan = 0; chan < channelCount; ++chan)
        {
            xplot_data_update(plotID[chan]);
            xplot_repaint(plotID[chan]);
            xplot_check_events(plotID[chan]);
        }
    }

    printf("%ld scans of %ld samples each measured.\n",
           scan, playbackBlocksize);

    while (waitWhenDone)
	for (chan = 0; chan < channelCount; chan++)
	    xplot_check_events(plotID[chan]);

    free(buf);
    free(dataBuf);
    free(plotID);

    return 0;
}


static void usage(const char* name)
{
    const char* basename;

    /* Get the program name in case the usage needs to be printed. */
    if ((basename = strrchr(name, '/')))
	++basename;
    else
	basename = name;

    fprintf(stderr, "Usage: %s [-abdhistTw] [volume:]filename\n"
	    "    The volume name is of the form 'vnm'.  The 'v' is just a\n"
	    "    literal 'v', the 'n' represents the SCSI address of a\n"
	    "    disk on the E1562's first SCSI bus, and the 'm' represents\n"
	    "    the SCSI address of a disk on the second SCSI bus.  Use\n"
	    "    'x' to specify no disk on one of the SCSI busses.\n"
	    "    -a             enable plot auto scale:      default %s\n"
            "    -b <blocksize> playback blocksize:          default %d\n"
            "    -c <channels>  channels to playback:        default ALL\n"
            "    -d <level>     debug level:                 default %d\n"
            "    -h             enable printing the header:  default OFF\n"
            "    -i <interface> interface name:              default %s\n"
            "    -s <count>     number of scans to playback: default %d\n"
	    "    -t             enable display of trailers:  default %d\n"
            "    -T <address>   E1562 logical address:       default %d\n"
	    "    -w             disable wait when done:      default %s\n"
            , basename,
	    DEFAULT_AUTOSCALE ? "ON" : "OFF",
            DEFAULT_BLOCKSIZE,
            DEFAULT_DEBUG_LEVEL,
            DEFAULT_INTERFACE,
            DEFAULT_NUMBER_SCANS,
	    DEFAULT_DISPLAY_TRAILERS,
            DEFAULT_E1562_LOGICAL_ADDRESS,
	    DEFAULT_WAIT_WHEN_DONE ? "ON" : "OFF");
}


typedef struct
{
    short called;
    short continueRecording;
    double bytes;
} CallbackArgs;

#if 0
static void doneCallback(E1562Handle e1562Handle, volatile void* arg,
                         double bytesRead)
{
    volatile CallbackArgs* myArg = (volatile CallbackArgs*)arg;

    if (myArg->continueRecording > 0)
    {
        (void) e1562tput_continueThroughput(e1562Handle);
        myArg->continueRecording = 0;
    }
    else
    {
        myArg->called = 1;
        myArg->bytes = bytesRead;
    }
}
#endif



int main(int argc, char** argv)
{
    int autoscale = DEFAULT_AUTOSCALE;
    long e1562LA = DEFAULT_E1562_LOGICAL_ADDRESS;
    char* interface = DEFAULT_INTERFACE;
    long numberScans = DEFAULT_NUMBER_SCANS;
    int display_trailers = DEFAULT_DISPLAY_TRAILERS;
    long blocksize = DEFAULT_BLOCKSIZE;
    long debugLevel = DEFAULT_DEBUG_LEVEL;
    char* channelString = "0:255";
    short enableHeaderPrint = 0;
    char* tputFile = 0;
    int waitWhenDone = DEFAULT_WAIT_WHEN_DONE;
    int opt;
    E1562Handle H1562;
    InputHwInformation inputSetup;
    size_t count;
    char fullFileName[e1562_FILENAME_MAX];
    long err;
    long index;
    unsigned long bytesPerScan;
    unsigned long bytesPerBlock;
    double totalPlaybackSamples;
    volatile CallbackArgs args;
    unsigned long recordScanCount;
    short channelList[MAX_MODULES * MAX_CHANNELS_PER_MODULE];
    short channelCount;

    /* Parse command-line arguments */
    while ((opt = getopt(argc, argv, "ab:c:d:his:tT:uw")) != EOF)
    {
        switch (opt)
        {
	case 'a':  autoscale = !autoscale; break;
        case 'b':  blocksize = atol(optarg); break;
        case 'c':  channelString = optarg; break;
        case 'd':  debugLevel = atol(optarg); break;
        case 'h':  enableHeaderPrint = !enableHeaderPrint; break;
        case 'i':  interface = optarg; break;
        case 's':  numberScans = atol(optarg); break;
	case 't':  display_trailers = !display_trailers; break;
        case 'T':  e1562LA = atol(optarg); break;
	case 'u':  usage(argv[0]); return 0;
	case '?':  usage(argv[0]); return 1;
	case 'w':  waitWhenDone = !waitWhenDone; break;
        }
    }

    /* Get the throughput filename.  This filename can include the volume
     * name of the LIF filesystem in addition to the filename by using a
     * colon to seperate the volume and file:  "Vx0:myFile".
     * If the volume is not specified, the program will try to determine a
     * default volume by looking for SCSI devices connected to the E1562.
     * If there is more than one disk connected to the E1562, this search
     * may not find the correct set of devices which specify the filesystem.
     * In these cases, the volume must be specified with the filename.
     */
    if (optind < argc && argv[optind] != 0)
    {
        tputFile = argv[optind++];
    }

    /* Make sure that a filename was specified. */
    if (tputFile == 0 || *tputFile == '\0')
    {
        fprintf(stderr, "Throughput playback file must be specified\n");
        usage(argv[0]);
        return 1;
    }

    /* Make sure that only one filename is specified. */
    if (optind < argc)
    {
        fprintf(stderr, "Extraneous arguments in command-line: ");
        for ( ; optind < argc; ++optind)
            fprintf(stderr, " %s", argv[optind]);
        fprintf(stderr, "\n");
        return 1;
    }


    parseAddressList(channelString, channelList, &channelCount,
                     MAX_MODULES * MAX_CHANNELS_PER_MODULE);
    if (channelCount == 0)
    {
        fprintf(stderr, "Channel list bad\n");
        return 1;
    }


    /* Initialize access to the E1562. */
    err = e1562tput_open(interface, (short) e1562LA, &H1562);
    if (err != 0)
    {
        printf("Can't open the E1562 module, error = %ld\n", err);
        return -1;
    }

    /* Open the specified file for playback.  This function will provide
     * a default volume name, but it may not be the correct volume, so it
     * is wise to specify the full volume:filename.
     */
    (void) strcpy(fullFileName, tputFile);
    err = e1562tput_openPlaybackFile(H1562, fullFileName);
    if (err != 0)
    {
        printf("Can't open throughput playback file, error = %ld\n", err);
        return -1;
    }

    /* Use a LIF library function to read the throughput header. */
    count = e1562tput_read(&inputSetup, sizeof(inputSetup), 1, H1562);
    if (count != 1)
    {
        printf("Can't read header information from throughput file\n");
        return -1;
    }

    if (channelCount > (short) inputSetup.activeChanCount)
        channelCount = inputSetup.activeChanCount;

    for (index = 0; index < channelCount; ++index)
    {
        if (channelList[index] < 0 ||
            channelList[index] > MAX_MODULES * MAX_CHANNELS_PER_MODULE)
        {
            fprintf(stderr, "%d is not valid channel number\n",
                    channelList[index]);
            return 1;
        }

        if (inputSetup.scaleFactor[channelList[index]] == 0.0)
        {
            fprintf(stderr, "Channel %d is not in the throughput file\n",
                    channelList[index]);
            return 1;
        }
    }

    /* Prepare the E1562 for playing back data from the file. */
    bytesPerBlock = dataSizeBytes(inputSetup.dataSize) *
        inputSetup.samplesPerBlock;
    if (inputSetup.appendStatusEnabled != 0)
        bytesPerBlock += sizeof(struct e1432_trailer);
    bytesPerScan = inputSetup.activeChanCount * bytesPerBlock;
    err = e1562tput_setupPlayback(H1562, inputSetup.dataOffset,
                                  bytesPerBlock);
    if (err != 0)
    {
        printf("Can't setup playback sequence\n");
        return -1;
    }

    recordScanCount = inputSetup.scanCount;
    if (inputSetup.firstScanOffset != 0) --recordScanCount;
    totalPlaybackSamples = (double)blocksize * (double)numberScans;
    if (totalPlaybackSamples > (double)inputSetup.samplesPerBlock *
        (double)recordScanCount)
    {
        totalPlaybackSamples =
            floor(((double)inputSetup.samplesPerBlock *
                   (double)recordScanCount) / blocksize) * blocksize;
    }

    args.called = 0;
    args.bytes = 0.0;
    args.continueRecording = (inputSetup.firstScanOffset != 0);

    /* Start the E1562 reading data into shared RAM.  This needs to be
     * synchronized with the function processing the data.  See the
     * function readPlaybackData for a better understanding of how the
     * data is acquired.
     */
    err = e1562tput_startPlayback(H1562, bytesPerScan * inputSetup.scanCount,
                                  (inputSetup.firstScanOffset /
                                   bytesPerScan) +
                                  args.continueRecording ? 1 : 0,
                                  0, 0);
    if (err != 0)
    {
        printf("Can't start playback sequence\n");
        return -1;
    }


    if (enableHeaderPrint) printHeader(fullFileName, &inputSetup);


    /* Read the desired amount of data. */
    err = readPlaybackData(&inputSetup, H1562,
                           (long)(totalPlaybackSamples / (double)blocksize),
                           blocksize, channelList, channelCount,
			   debugLevel, autoscale, waitWhenDone,
			   display_trailers);
    if (err != 0)
    {
        printf("Can't read playback data:  error = %ld\n", err);
    }

    err = e1562tput_abortThroughput(H1562);
    if (err != 0)
    {
        printf("Can't abort the throughput playback, error = %ld\n", err);
        return -1;
    }

    /* Cleanup the LIF library and close all SCSI devices on the E1562. */
    err = e1562tput_close(H1562);
    if (err != 0)
    {
        printf("Can't close the E1562, error = %ld\n", err);
        return -1;
    }

    return 0;
}
